/* cfc -- C function caller. */

/*  Copyright 2006 Miller Puckette; available on the BSD license:
    http://crca.ucsd.edu/~msp/Software/LICENSE.txt
*/
#include <string.h>
#include <stdio.h>
#ifdef MSW
#include <windows.h>
int close( int);
#else
#include <unistd.h>
#include <dlfcn.h>
#endif

typedef int (*t_cfcfn)(double f0, double f1, double f2, double f3, 
    double f4, double f5, double f6, double f7, double f8, double f9);

static t_cfcfn cfc_doload(char *filename, char *funcname, void **dllobjp,
    char *errorstring, int estringlength)
{
    char altfuncname[1000];
    t_cfcfn fn;
    void *dlobj = 0;
#ifdef MSW
    HINSTANCE ntdll;
#endif
    strcpy(altfuncname, "_");
    strncat(altfuncname, funcname, 998);
    altfuncname[999] = 0;
#ifdef MSW
    ntdll = LoadLibrary(filename);
    if (!ntdll)
    {
        if (estringlength >= 80)
            strcat(errorstring, "couldn't load object file");
        else errorstring[0] = 0;
        *dllobjp = 0;
        return (0);
    }
    dlobj = (void *)ntdll;
    fn = (t_cfcfn)GetProcAddress(ntdll, funcname);  
    if (!fn)
        fn = (t_cfcfn)GetProcAddress(ntdll, altfuncname);  
#else
    dlobj = dlopen(filename, RTLD_NOW | RTLD_GLOBAL);
    if (!dlobj)
    {
        strncpy(errorstring, dlerror(), estringlength);
        errorstring[estringlength-1] = 0;
        *dllobjp = 0;
        return (0);
    }
    fn = (t_cfcfn)dlsym(dlobj,  funcname);
    if (!fn)
        fn = (t_cfcfn)dlsym(dlobj,  altfuncname);
    if (!fn)
    {
        if (estringlength >= 80)
            sprintf(errorstring, "%.40s: symbol not found in object file",
                funcname);
        else errorstring[0] = 0;
        if (dlobj)
        {
#ifdef MSW
            ... free it...
#else
            dlclose(dlobj);
#endif
            dlobj = 00;
        }    
    }
#endif
    *dllobjp = dlobj;
    return (fn);
}


#include "m_pd.h"
static t_class *cfc_class;
#define MAXIO 10

typedef struct _cfc
{
    t_object x_obj;
    t_canvas *x_canvas;
    t_symbol *x_filesym;
    int x_nin;
    int x_nout;
    t_outlet *x_outlet[MAXIO];
    float x_inval[MAXIO];
    t_cfcfn x_fn;
    void *x_code;
} t_cfc;

static void cfc_load(t_cfc *x, t_symbol *s, t_symbol *s2);

static void *cfc_new(t_symbol *s, t_symbol *s2, t_float fnin, t_float fnout)
{
    t_cfc *x = (t_cfc *)pd_new(cfc_class);
    int i, nin = fnin, nout = fnout;
    
    if (nin > 10)
        nin = 10;
    else if (nin < 1)
        nin = 1;
    if (nout > 10)
        nout = 10;
    else if (nout < 0)
        nout = 0;
    x->x_nin = nin;
    x->x_nout = nout;
    for (i = 0; i < nout; i++)
        x->x_outlet[i] = outlet_new(&x->x_obj, &s_float);
    for (i = 0; i < MAXIO; i++)
         x->x_inval[i] = 0;
    for (i = 1; i < nin; i++)
         floatinlet_new(&x->x_obj, &x->x_inval[i]);
    x->x_canvas = canvas_getcurrent();
    x->x_fn = 0;
    x->x_code = 0;
    if (*s->s_name && *s2->s_name)
        cfc_load(x, s, s2);
    return (x);
}

static void cfc_unload(t_cfc *x)
{
    if (x->x_code)
#ifdef MSW
        FreeLibrary(x->x_code);
#else
        dlclose(x->x_code);
#endif
    x->x_code = 0;
    x->x_fn = 0;
}

static void cfc_load(t_cfc *x, t_symbol *s, t_symbol *s2)
{
    char estring[200];
    char buf[MAXPDSTRING], *bufptr, filename[MAXPDSTRING];
    int fd = open_via_path(canvas_getdir(x->x_canvas)->s_name,
         s->s_name, "", buf, &bufptr, MAXPDSTRING, 0);
    if (fd < 0)
    {
        pd_error(x, "%s: can't open file", s->s_name);
        return;
    }
    close(fd);
    strncpy(filename, buf, MAXPDSTRING);
    filename[MAXPDSTRING-2] = 0;
    strcat(filename, "/");
    strncat(filename, bufptr, MAXPDSTRING-strlen(filename));
    filename[MAXPDSTRING-1] = 0;
    sys_bashfilename(filename, filename);
    cfc_unload(x);
    x->x_fn = cfc_doload(filename, s2->s_name, &x->x_code, estring, 200);
    if (!x->x_fn)
    {
        pd_error(x, "can't load function or can't find symbol");
        post("%s", estring);
    }
}

static t_cfc *cfc_current;

#ifdef MSW
__declspec(dllexport) extern
#endif
void cfc_sendoutlet(int which, double val)
{
    if (cfc_current)
    {
        if (which >= 1 && which <= cfc_current->x_nout)
            outlet_float(cfc_current->x_outlet[which-1], val);
        else pd_error(cfc_current,
            "cfc: outlet number %d out of range", which);
    }
    else error("cfc: no current object for outlet");
} 

static void cfc_bang(t_cfc *x)
{
    t_cfc *was = cfc_current;
    cfc_current = x;
    if (x->x_fn)
        (*x->x_fn)(x->x_inval[0], x->x_inval[1], x->x_inval[2], 
            x->x_inval[3], x->x_inval[4], x->x_inval[5], x->x_inval[6],
            x->x_inval[7], x->x_inval[8], x->x_inval[9]); 
    else pd_error(x, "cfc: no function loaded");
    cfc_current = was;
}

static void cfc_float(t_cfc *x, t_floatarg f)
{
    x->x_inval[0] = f;
    cfc_bang(x);
}

static void cfc_free(t_cfc *x)
{
    cfc_unload(x);
    cfc_current = 0;
}

void cfc_setup(void)
{
    cfc_class = class_new(gensym("cfc"), (t_newmethod)cfc_new,
        (t_method)cfc_free, sizeof(t_cfc), 0, A_DEFSYMBOL, A_DEFSYMBOL, 
            A_DEFFLOAT, A_DEFFLOAT, 0);
    class_addmethod(cfc_class, (t_method)cfc_load, gensym("load"), 
        A_SYMBOL, A_SYMBOL, 0);
    class_addmethod(cfc_class, (t_method)cfc_unload, gensym("unload"), 0);
    class_addbang(cfc_class, cfc_bang);
    class_addfloat(cfc_class, cfc_float);
}
